import Joi from 'joi';
import {
  shopEndPoints, applicationChargeEndpoint, productsEndpoints, webhooksEndpoint, streamPageEndpoints, headers,
} from '../constants/index.js';
import {
  notAcceptable, extractJoiErrorMessage,
  authorizeInstanceIdAndMasterKey,
  errorHandler, getHeaders, notFound, authorizeWebhooks, authorizeInstanceId,
} from '../helper/index.js';

const masterKeyAuthReqEndpoints = [
  shopEndPoints.SHOP_ENDPOINT,
  applicationChargeEndpoint.RECURRING_CHARGE_ENDPOINT,
  applicationChargeEndpoint.APP_INSTANCE_ENDPOINS,
];

const authorizeRequiredForInstanceId = [
  streamPageEndpoints.GET_CHANNELIZE_DATA,
  productsEndpoints.GET_PRODUCTS_ENDPOINT,
];

const wixWebhooksEndpoints = [
  webhooksEndpoint.CUSTOMER_CREATE_ENDPOINT,
  webhooksEndpoint.CUSTOMER_UPDATE_ENDPOINT,
  webhooksEndpoint.CUSTOMER_DELETE_ENDPOINT,
  webhooksEndpoint.APP_UNINSTALL_ENDPOINT,
  webhooksEndpoint.PLAN_AUTORENEWAL_CANCELLED_ENDPOINTS,
];
const getEndpoints = (url) => {
  if (url.includes('?')) {
    return url.split('?')[0];
  }
  return url;
};
export const authMiddleware = async (ctx, next) => {
  try {
    const url = getEndpoints(ctx.request.url);
    if (masterKeyAuthReqEndpoints.includes(url)) {
      validateAuthHeader(ctx);
      const { instanceId, chMasterKey } = getHeaders(ctx);
      if (instanceId && chMasterKey) {
        const shopData = await authorizeInstanceIdAndMasterKey(instanceId, chMasterKey);
        ctx.request.shopData = shopData;
      } else {
        throw notFound('headers are not valid');
      }
    } else if (authorizeRequiredForInstanceId.includes(url)) {
      const { instanceId } = getHeaders(ctx);
      if (instanceId) {
        const shopData = await authorizeInstanceId(instanceId);
        ctx.request.shopData = shopData;
      }
    } else if (wixWebhooksEndpoints.includes(url)) {
      const token = ctx.request.body;
      if (token) {
        const payload = await authorizeWebhooks(token);
        ctx.request.payload = JSON.parse(payload.data);
      } else {
        throw notFound('webhooks are not verified');
      }
    }
    await next();
  } catch (error) {
    errorHandler(ctx, error);
  }
};

const validateAuthHeader = (ctx) => {
  const authValidatorSchema = Joi.object({
    [headers.WIX_INSTANCE_ID_HEADER]: Joi.string().required().label(headers.WIX_INSTANCE_ID_HEADER),
    [headers.CH_MASTER_KEY_HEADER_NAME]: Joi.string().required().label(headers.CH_MASTER_KEY_HEADER_NAME),
  });
  const headerError = authValidatorSchema.validate(ctx.headers, { allowUnknown: true }).error;
  if (headerError) {
    const errorMessage = extractJoiErrorMessage(headerError);
    throw notAcceptable(errorMessage);
  }
};
