import { ShopModel, WixTokenModel } from '../models/index.js';
import { Collections } from '../constants/index.js';

const models = {
  [Collections.SHOPS]: ShopModel,
  [Collections.WIX_TOKENS]: WixTokenModel,
};

const shopsSchemaKeysMappings = {
  shopId: '_id',
  instanceId: 'instance_id',
  chApplicationId: 'ch_application_id',
  chPublicKey: 'ch_public_key',
  chPrivateKey: 'ch_private_key',
  chCustomerId: 'ch_customer_id',
  isAppUninstalled: 'is_app_uninstalled',
  createdAt: 'created_at',
  updatedAt: 'updated_at',
};

const wixTokensSchemaMappings = {
  wixTokenId: '_id',
  shopIdFk: 'shop_id_fk',
  refreshToken: 'refresh_token',
  createdAt: 'created_at',
  updatedAt: 'updated_at',
};

const tableSchemaMapping = {
  [Collections.SHOPS]: shopsSchemaKeysMappings,
  [Collections.WIX_TOKENS]: wixTokensSchemaMappings,
};

export const create = (collectionName, data) => {
  const schemaKeysMappings = tableSchemaMapping[collectionName];
  const dataObj = getSchemaMappedObj(schemaKeysMappings, data);
  const modelInstance = new models[collectionName](dataObj);
  return modelInstance.save();
};

export const findOne = (collectionName, queryData, fields) => {
  const schemaKeysMappings = tableSchemaMapping[collectionName];
  const queryObj = getSchemaMappedObj(schemaKeysMappings, queryData);
  const selectFieldObj = getRequiredFields(schemaKeysMappings, fields);
  return models[collectionName].findOne(queryObj, selectFieldObj);
};

export const findAll = (collectionName, queryData, fields) => {
  const schemaKeysMappings = tableSchemaMapping[collectionName];
  const queryObj = getSchemaMappedObj(schemaKeysMappings, queryData);
  const selectFieldObj = getRequiredFields(schemaKeysMappings, fields);
  return models[collectionName].find(queryObj, selectFieldObj);
};

export const updateOne = (collectionName, queryData, updateData) => {
  const schemaKeysMappings = tableSchemaMapping[collectionName];
  const queryMappedData = getSchemaMappedObj(schemaKeysMappings, queryData);
  const updateMappedData = getSchemaMappedObj(schemaKeysMappings, updateData);
  return models[collectionName].updateOne(queryMappedData, updateMappedData);
};

export const updateAll = (collectionName, queryData, updateData) => {
  const schemaKeysMappings = tableSchemaMapping[collectionName];
  const queryMappedData = getSchemaMappedObj(schemaKeysMappings, queryData);
  const updateMappedData = getSchemaMappedObj(schemaKeysMappings, updateData);
  return models[collectionName].updateMany(queryMappedData, updateMappedData);
};

export const deleteOne = (collectionName, queryData) => {
  const schemaKeysMappings = tableSchemaMapping[collectionName];
  const queryMappedData = getSchemaMappedObj(schemaKeysMappings, queryData);
  return models[collectionName].deleteOne(queryMappedData);
};

export const findOneAndDelete = (collectionName, queryData) => {
  const schemaKeysMappings = tableSchemaMapping[collectionName];
  const queryMappedData = getSchemaMappedObj(schemaKeysMappings, queryData);
  return models[collectionName].findOneAndDelete(queryMappedData);
};

const getSchemaMappedObj = (schemaKeysMappings, queryData) => {
  const queryObj = {};
  Object.keys(queryData).forEach((key) => {
    const schemaKey = schemaKeysMappings[key];
    queryObj[schemaKey] = queryData[key];
  });
  return queryObj;
};

const getRequiredFields = (schemaKeysMappings, fields = []) => {
  const selectFieldObj = {};
  fields.forEach((field) => {
    const schemaKey = schemaKeysMappings[field];
    selectFieldObj[schemaKey] = 1;
  });
  return selectFieldObj;
};
