import { Collections, channelizeWebsiteEndpoint } from '../constants/index.js';
import {
  updateOne, findOne, deleteOne, createChannelizeHeader, postApi, putApi, deleteApi, logger
} from '../helper/index.js';

export const createdCustomerHandler = async (ctx) => {
  logger.debug('Entering: createdCustomerHandler');
  const { payload } = ctx.request;
  const instanceId = payload.instanceId;
  const customerData = JSON.parse(payload.data);

  const id = customerData.entityId;
  const first_name = customerData.createdEvent.entity.contact.firstName;
  const last_name = customerData.createdEvent.entity.contact.lastName;

  const shopData = await findOne(Collections.SHOPS, { instanceId }, ['chPrivateKey']);
  const { ch_private_key: chPrivateKey } = shopData || {};
  if (!chPrivateKey) {
    logger.error('error in webhook createdCustomerHandler. shop data not found');
    return null;
  }
  const body = {
    id,
    displayName: `${first_name} ${last_name}`,
  };
  return createCustomerOnChannelize(chPrivateKey, body);
};

export const createCustomerOnChannelize = async (chPrivateKey, body) => {
  const channelizeCreateUserUrl = `${process.env.CHANNELIZE_CHAT_API_HOST}/users`;
  const headers = createChannelizeHeader(chPrivateKey);
  const createdCustChannelizeRes = await postApi(channelizeCreateUserUrl, { headers }, body);
  if (createdCustChannelizeRes.response && createdCustChannelizeRes.response.data) {
    logger.error('error channelize create user url:', createdCustChannelizeRes.response.data);
    return null;
  }
  return createdCustChannelizeRes.data;
};

export const updatedCustomerHandler = async (ctx) => {
  logger.debug('Entering: updatedCustomerHandler');
  const { payload } = ctx.request;
  const instanceId = payload.instanceId;
  const customerData = JSON.parse(payload.data);

  const id = customerData.entityId;
  const first_name = customerData.updatedEvent.currentEntity.contact.firstName;
  const last_name = customerData.updatedEvent.currentEntity.contact.lastName;
  const channelizeUpdateUserUrl = `${process.env.CHANNELIZE_CHAT_API_HOST}/users/${id}`;
  const shopData = await findOne(Collections.SHOPS, { instanceId }, ['chPrivateKey']);
  const { ch_private_key: chPrivateKey } = shopData || {};
  if (!chPrivateKey) {
    logger.error('Error in webhook updatedCustomerHandler. shop data not exits');
    return null;
  }
  const headers = createChannelizeHeader(chPrivateKey);
  const body = {
    displayName: `${first_name} ${last_name}`,
  };
  const updatedCustChannelizeRes = await putApi(channelizeUpdateUserUrl, { headers }, body);
  if (updatedCustChannelizeRes.response && updatedCustChannelizeRes.response.data) {
    logger.error('error channelize update user url: ', updatedCustChannelizeRes.response.data);
    return null;
  }
  return updatedCustChannelizeRes.data;
};

export const deletedCustomerHandler = async (ctx) => {
  logger.debug('Entering: deletedCustomerHandler');
  const { payload } = ctx.request;
  const instanceId = payload.instanceId;

  const customerData = JSON.parse(payload.data);
  const id = customerData.entityId;
  const channelizeDeleteUserUrl = `${process.env.CHANNELIZE_CHAT_API_HOST}/users/${id}`;
  const shopData = await findOne(Collections.SHOPS, { instanceId }, ['chPrivateKey']);
  const { ch_private_key: chPrivateKey } = shopData || {};
  if (!chPrivateKey) {
    logger.error('Error in webhook deletedCustomerHandler. shop data not exits');
    return null;
  }
  const headers = createChannelizeHeader(chPrivateKey);
  const deleteCustChannelizeRes = await deleteApi(channelizeDeleteUserUrl, { headers });
  if (deleteCustChannelizeRes.response && deleteCustChannelizeRes.response.data) {
    logger.error(`error channelize delete user url: ${deleteCustChannelizeRes.response.data}`);
    return null;
  }
  return deleteCustChannelizeRes.data;
};

export const appUninstallHandler = async (ctx) => {
  logger.debug('Entering: appUninstallHandler');
  const { payload } = ctx.request;
  const instanceId = payload.instanceId;
  await updateOne(Collections.SHOPS, { instanceId }, { isAppUninstalled: true });
  const updatedShopData = await findOne(Collections.SHOPS, { instanceId });
  if (!updatedShopData) {
    logger.error('error in webhhook appUninstallHandler. shop data not found');
    return null;
  }
  const { _id: shopIdFk, ch_private_key: chPrivateKey } = updatedShopData;
  await deleteOne(Collections.WIX_TOKENS, { shopIdFk });
  await disableChannelizeApplication(chPrivateKey);

  return 'data deleted succesfully';
};

const disableChannelizeApplication = async (chPrivateKey) => {
  const privateKeyBuffer = new Buffer.from(chPrivateKey);
  const body = {
    privateKey: privateKeyBuffer.toString('base64'),
  };
  const url = `${process.env.CHANNELIZE_WEBSITE_HOST}${channelizeWebsiteEndpoint.SUSPEND_APPLICATION}`;
  return postApi(url, { 'Content-Type': 'application/json' }, body);
};

export const paidPlanAutoRenewalCancelled = async (ctx) => {
  logger.debug('Entering: paidPlanAutoRenewalCancelled');
  const { payload } = ctx.request;
  const instanceId = payload.instanceId;
  const shopData = await findOne(Collections.SHOPS, { instanceId }, ['chPrivateKey']);
  const { ch_private_key: chPrivateKey } = shopData || {};
  if (!chPrivateKey) {
    logger.error('error in webhook paidPlanAutoRenewalCancelled. shop data not found');
    return null;
  }
  await paidPlanAutoRenewalCancelledApplication(chPrivateKey);

  return 'Paid Plan Auto Renewal Cancelled Successfully';
};

const paidPlanAutoRenewalCancelledApplication = async (chPrivateKey) => {
  const privateKeyBuffer = new Buffer.from(chPrivateKey);
  const body = {
    privateKey: privateKeyBuffer.toString('base64'),
  };
  const url = `${process.env.CHANNELIZE_WEBSITE_HOST}${channelizeWebsiteEndpoint.PLAN_CANCELLED_APPLICATION}`;
  return postApi(url, { 'Content-Type': 'application/json' }, body);
};
