import { get } from 'lodash';
import {
  unauthorized, notFound, getHeaders, createChannelizeHeader, getApi, getAccessToken, failedDependency, postApi, logger
} from '../helper/index.js';
import { createCustomerOnChannelize } from './index.js';
import { wixGetCustomerDataById } from '../wix/resources/index.js';

export const getChannelizeData = async (ctx) => {
  try {
    logger.debug('Entering getChannelizeData controller:');
    const { wixUserId = null } = getHeaders(ctx);
    const { ch_public_key: chPublicKey, _id: shopId, ch_private_key: chPrivateKey } = ctx.request.shopData;
    if (!chPublicKey) {
      throw notFound('public key not found');
    }
    const response = {};
    response.shopData = {
      _id: shopId,
      ch_public_key: chPublicKey,
    };
    if (!wixUserId) {
      return response;
    }

    const customerData = await getChannelizeCustomerData(chPrivateKey, wixUserId);
    if (!customerData) { // Case if customer not created on channelize
      logger.debug('user not created on channelize.');
      const wixAccessToken = await getAccessToken(shopId);
      const wixCustomerData = await wixGetCustomerDataById(wixUserId, wixAccessToken);
      if (wixCustomerData.response && wixCustomerData.response.data) {
        logger.error('not geting customer data on wix');
        throw notFound('not geting customer data on wix');
      }

      const customer = get(wixCustomerData, 'data.member.contact');
      const body = {
        id: wixUserId,
        displayName: `${customer.firstName} ${customer.lastName}`,
      };
      const createdUserData = await createCustomerOnChannelize(chPrivateKey, body);
      if (!createdUserData) {
        logger.error('user not created on channelize');
        throw failedDependency('user not created on channelize');
      }
    }
    const channelizeTokenResponse = await getChannelizeToken(wixUserId, chPrivateKey);
    if (channelizeTokenResponse && channelizeTokenResponse.response) {
      const error = get(channelizeTokenResponse, 'response.data', null);
      logger.error('error on getting token from channelize', { messageObject: error });
      throw failedDependency('error on getting token from channelize');
    }
    const channelizeTokenData = get(channelizeTokenResponse, 'data', {});
    response.token = channelizeTokenData;

    return response;
  } catch (error) {
    logger.error('getChannelizeData controller error: ', { messageObject: error });
    throw unauthorized('getChannelizeData is not authorized');
  }
};

const getChannelizeToken = (wixUserId, chPrivateKey) => {
  logger.debug(`Entering: getChannelizeToken: wixUserId: ${wixUserId}`);
  const headers = createChannelizeHeader(chPrivateKey);
  const body = {
    userId: wixUserId,
  };
  const url = `${process.env.CHANNELIZE_CHAT_API_HOST}/users/create_access_token`;
  return postApi(url, { headers }, body);
};

export const getChannelizeCustomerData = async (chPrivateKey, wixUserId) => {
  const headers = createChannelizeHeader(chPrivateKey);
  const channelizeExportUserUrl = `${process.env.CHANNELIZE_CHAT_API_HOST}/users/${wixUserId}/export`;
  const customerChannelizeRes = await getApi(channelizeExportUserUrl, { headers });
  if (customerChannelizeRes.response && customerChannelizeRes.response.data) {
    logger.error(`error channelize get customer data: ${customerChannelizeRes.response.data}`);
    return null;
  }
  return customerChannelizeRes.data;
};
