import {
  updateOne, findOne, create, notFound, unauthorized, getDashboardUrl, logger
} from '../helper/index.js';
import { Collections } from '../constants/index.js';
import { wixGetRefreshToken } from '../wix/resources/index.js';

export const getWixInstallUrl = async (ctx) => {
  try {
    const { WIX_URL: url, APP_ID: appId, HOST: redirectUrl } = process.env;
    const { token } = ctx.request.query;
    return (`${url}/installer/install?token=${token}&appId=${appId}&redirectUrl=${redirectUrl}/auth`);
  } catch (error) {
    logger.error('Error in installing process: ', { messageObject: error });
  }
};

export const registerShop = async (ctx) => {
  try {
    const { code, instanceId } = ctx.request.query;
    const wixTokens = await wixGetRefreshToken(code);
    if (!wixTokens) {
      throw unauthorized('Refresh Token in found');
    }
    const refreshToken = wixTokens.data.refresh_token;
    if (!refreshToken) {
      throw notFound('Refresh Token not found');
    }
    // register shop if instanceId is not found
    const shopResponse = await findOne(Collections.SHOPS, { instanceId });
    if (shopResponse) {
      await updateOne(Collections.SHOPS, { instanceId }, { isAppUninstalled: false });
      const shopIdFk = shopResponse._id;
      const wixTokenCollectionData = await findOne(Collections.WIX_TOKENS, { shopIdFk });
      if (wixTokenCollectionData) {
        await updateOne(Collections.WIX_TOKENS, { shopIdFk }, { refreshToken });
      } else {
        await create(Collections.WIX_TOKENS, { shopIdFk, refreshToken });
      }
      return shopIdFk;
    }

    const wixShopCollection = await create(Collections.SHOPS, { instanceId });
    const shopIdFk = wixShopCollection._id;
    // register wixToken Collection
    await create(Collections.WIX_TOKENS, { shopIdFk, refreshToken });
    return shopIdFk;
  } catch (error) {
    logger.error('Error in auth installing process: ', { messageObject: error });
    throw unauthorized('shop error');
  }
};

export const getRedirectUrl = async (ctx) => {
  try {
    const { instance } = ctx.request.query;
    var pair = instance.split('.');
    var data = pair[1];
    const bufferData = JSON.parse(Buffer.from(data, 'base64').toString('ascii'));
    const instanceId = bufferData.instanceId;
    if (!instanceId) {
      throw notFound('data not found');
    }
    return instanceId;
  } catch (error) {
    logger.error('Error in redirecting process', { messageObject: error });
    throw unauthorized('Error in redirecting app');
  }
};

export const getPricingPage = async (ctx) => {
  try {
    const { instance } = ctx.request.query;
    if (!instance) {
      const { CHANNELIZE_WEBSITE_HOST, WIX_URL } = process.env;
      return `https://pricing.channelize.io/plans?continue=${WIX_URL}/app-market/live-shopping-by-channelize-io&source=wix`;
    }
    var pair = instance.split('.');
    var data = pair[1];
    const bufferData = JSON.parse(Buffer.from(data, 'base64').toString('ascii'));
    const instanceId = bufferData.instanceId;
    if (!instanceId) {
      throw notFound('data not found');
    }
    const redirectUrl = getDashboardUrl(instanceId);
    return redirectUrl;
  } catch (error) {
    logger.error('Error in redirecting process', { messageObject: error });
    throw unauthorized('error in getPricing Page');
  }
};
